<!DOCTYPE html>
<html lang="pt-BR">

<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>MEI Regulariza - Consulta CNPJ</title>
    <script src="https://unpkg.com/vue@3/dist/vue.global.js"></script>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="css/custom.css">
    <script src="components/Layout.js"></script>
    <script src="components/Icon.js"></script>
    <style>
        @keyframes spin {
            to { transform: rotate(360deg); }
        }
        .animate-spin {
            animation: spin 1s linear infinite;
        }
        /* Melhorias para mobile */
        .touch-manipulation {
            touch-action: manipulation;
            -webkit-tap-highlight-color: transparent;
        }
        /* Ajuste de input para mobile - evita zoom no iOS */
        @media screen and (max-width: 768px) {
            input[type="text"],
            input[type="tel"],
            input[type="email"] {
                font-size: 16px !important;
            }
        }
        /* Melhorias de UX/UI */
        body {
            overflow-x: hidden;
        }
        .bg-card {
            transition: box-shadow 0.3s ease;
        }
        .bg-card:hover {
            box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06);
        }
    </style>
</head>

<body class="bg-background text-foreground min-h-screen">
    <div id="app" class="min-h-screen flex flex-col">
        <layout-component>
            <div class="w-full px-4 sm:px-6 md:px-8 pt-12 sm:pt-16 md:pt-20 pb-4 sm:pb-6">
                <div class="max-w-2xl mx-auto bg-card rounded-lg shadow-sm border border-border p-5 sm:p-6 md:p-8 mb-8" style="margin-top: 100px;">
                    <!-- Cabeçalho -->
                    <div class="text-center mb-5 sm:mb-8">
                        <div class="flex justify-center mb-3 sm:mb-4">
                            <img src="assets/govbr.webp" alt="gov.br" class="h-10 sm:h-12 w-auto">
                        </div>
                        <h1 class="text-lg sm:text-2xl font-bold text-foreground mb-2">
                            Consulta de CNPJ MEI
                        </h1>
                        <p class="text-muted-foreground text-xs sm:text-sm">
                            Digite o CNPJ para verificar a situação dos seus débitos DAS
                        </p>
                    </div>

                    <!-- Formulário -->
                    <form @submit.prevent="consultarCNPJ" class="space-y-5 sm:space-y-6">
                        <!-- Campo CNPJ -->
                        <div>
                            <label for="cnpj" class="block text-sm font-medium text-foreground mb-2">
                                CNPJ <span class="text-destructive">*</span>
                            </label>
                            <input
                                type="text"
                                id="cnpj"
                                v-model="cnpj"
                                @input="formatarCNPJ"
                                placeholder="00.000.000/0000-00"
                                maxlength="18"
                                required
                                class="w-full px-4 py-3.5 text-base border border-border rounded-md focus:outline-none focus:ring-2 focus:ring-primary focus:border-transparent text-foreground bg-white"
                                :class="{ 'border-destructive': erroCNPJ }"
                            >
                            <p v-if="erroCNPJ" class="mt-2 text-sm text-destructive">
                                {{ erroCNPJ }}
                            </p>
                            <p class="mt-2 text-xs text-muted-foreground">
                                Digite apenas números ou use a formatação automática
                            </p>
                        </div>

                        <!-- Botão Consultar -->
                        <button
                            type="submit"
                            :disabled="consultando || !cnpjValido"
                            class="w-full text-white font-semibold text-base px-6 py-4 rounded-md hover:opacity-90 active:opacity-75 transition-opacity disabled:opacity-50 disabled:cursor-not-allowed touch-manipulation"
                            style="background-color: #7CB342; min-height: 52px;"
                        >
                            <span v-if="consultando" class="flex items-center justify-center gap-2">
                                <icon-component name="spinner" size="h-5 w-5" color="text-white" class="animate-spin"></icon-component>
                                <span>Consultando...</span>
                            </span>
                            <span v-else>
                                Consultar CNPJ
                            </span>
                        </button>
                    </form>

                    <!-- Informações -->
                    <div class="mt-6 sm:mt-8 pt-5 sm:pt-6 border-t border-border">
                        <div class="bg-blue-50 border border-blue-200 rounded-lg p-4">
                            <div class="flex items-start gap-3">
                                <icon-component name="shield" size="h-5 w-5" color="text-primary" class="flex-shrink-0 mt-0.5"></icon-component>
                                <div class="text-sm text-foreground">
                                    <p class="font-semibold mb-1">Segurança dos seus dados</p>
                                    <p class="text-muted-foreground leading-relaxed text-xs sm:text-sm">
                                        Suas informações são tratadas com total segurança e sigilo, conforme a Lei Geral de Proteção de Dados (LGPD).
                                    </p>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Rodapé de Informações -->
                <div class="mt-8 sm:mt-10 text-center text-xs sm:text-sm text-muted-foreground px-2 max-w-2xl mx-auto">
                    <p class="leading-relaxed">
                        Sistema oficial de consulta e regularização de débitos DAS para Microempreendedores Individuais
                    </p>
                </div>
            </div>
        </layout-component>
    </div>

    <script>
        const { createApp } = Vue;

        createApp({
            components: {
                'layout-component': window.LayoutComponent,
                'icon-component': window.IconComponent
            },
            data() {
                return {
                    cnpj: '',
                    consultando: false,
                    erroCNPJ: ''
                }
            },
            computed: {
                cnpjValido() {
                    const apenasNumeros = this.cnpj.replace(/\D/g, '');
                    return apenasNumeros.length === 14;
                }
            },
            methods: {
                formatarCNPJ(event) {
                    let valor = event.target.value.replace(/\D/g, '');
                    
                    // Limita a 14 dígitos
                    if (valor.length > 14) {
                        valor = valor.substring(0, 14);
                    }
                    
                    // Formata: 00.000.000/0000-00
                    if (valor.length <= 2) {
                        this.cnpj = valor;
                    } else if (valor.length <= 5) {
                        this.cnpj = valor.replace(/(\d{2})(\d+)/, '$1.$2');
                    } else if (valor.length <= 8) {
                        this.cnpj = valor.replace(/(\d{2})(\d{3})(\d+)/, '$1.$2.$3');
                    } else if (valor.length <= 12) {
                        this.cnpj = valor.replace(/(\d{2})(\d{3})(\d{3})(\d+)/, '$1.$2.$3/$4');
                    } else {
                        this.cnpj = valor.replace(/(\d{2})(\d{3})(\d{3})(\d{4})(\d+)/, '$1.$2.$3/$4-$5');
                    }
                    
                    // Limpa erro quando começar a digitar
                    if (this.erroCNPJ) {
                        this.erroCNPJ = '';
                    }
                },
                
                validarCNPJ() {
                    const apenasNumeros = this.cnpj.replace(/\D/g, '');
                    
                    if (apenasNumeros.length === 0) {
                        this.erroCNPJ = 'Por favor, digite o CNPJ';
                        return false;
                    }
                    
                    if (apenasNumeros.length !== 14) {
                        this.erroCNPJ = 'CNPJ deve conter exatamente 14 dígitos';
                        return false;
                    }
                    
                    // Validação básica de dígitos verificadores (opcional)
                    // Aqui você pode adicionar validação mais robusta se necessário
                    
                    this.erroCNPJ = '';
                    return true;
                },
                
                consultarCNPJ() {
                    if (!this.validarCNPJ()) {
                        return;
                    }
                    
                    this.consultando = true;
                    const apenasNumeros = this.cnpj.replace(/\D/g, '');
                    
                    // Limpa dados anteriores do localStorage
                    localStorage.removeItem('meiData');
                    localStorage.removeItem('currentTransaction');
                    
                    // Redireciona para a.php com o CNPJ
                    window.location.href = `a.php?cnpj=${apenasNumeros}`;
                }
            },
            mounted() {
                // Foca no campo CNPJ ao carregar
                this.$nextTick(() => {
                    const input = document.getElementById('cnpj');
                    if (input) {
                        input.focus();
                    }
                });
            }
        }).mount('#app');
    </script>
</body>

</html>
